import { Nullable } from "./Maybe";
/**
 * Type guard that checks if a value is a non-null, non-array object.
 *
 * @param obj - The value to check
 * @returns `true` if `obj` is a plain object (not null, not an array)
 */
export declare function isObject(obj: unknown): obj is object;
/**
 * Returns an array of own enumerable string keys from an object.
 *
 * Unlike `Object.keys`, this filters to only own enumerable properties and
 * returns an empty array for null/undefined inputs.
 *
 * @param o - The object to extract keys from
 * @returns Array of own enumerable string keys, or empty array if `o` is nullish
 */
export declare function keys<T extends object, K extends string & keyof T>(o: T): K[];
/**
 * Type guard that checks if a value is a function.
 *
 * @param obj - The value to check
 * @returns `true` if `obj` is a function
 */
export declare function isFunction(obj: unknown): obj is (...args: unknown[]) => unknown;
/**
 * Turns an array of `[key, value]` pairs into an object.
 *
 *  - Pairs whose key is `null | undefined` **or** value is `undefined` are skipped.
 *  - If `base` is provided it is mutated and returned (handy for “extend” use‑cases).
 */
export declare function fromEntries<K extends PropertyKey, V = unknown>(pairs: Nullable<Nullable<[Nullable<K>, V]>[]>, base?: Record<K, V>): Record<K, V>;
/**
 * Utility type that removes specified keys from a type by setting their values to `never`.
 *
 * @typeParam T - The source object type
 * @typeParam U - The keys to exclude
 */
export type Unpick<T, U> = {
    [P in keyof T]: P extends U ? never : T[P];
};
/**
 * Returns a shallow copy of an object with the specified keys omitted.
 *
 * @param t - The source object
 * @param keysToOmit - Keys to exclude from the result
 * @returns A new object without the specified keys
 */
export declare function omit<T extends object, S extends string>(t: T, ...keysToOmit: S[]): Unpick<T, S>;
/**
 * Provides a type-safe exhaustive array of keys for a given interface.
 *
 * Unfortunately, `satisfies (keyof T)[]` doesn't ensure all keys are present,
 * and doesn't guard against duplicates. This function does.
 *
 * @param t - The interface to extract keys from. This is a Record of keys to
 * `true`, which ensures the returned key array is unique.
 */
export declare function keysOf<T>(t: Required<Record<keyof T, true>>): (keyof T)[];
