import child_process from "node:child_process";
/**
 * These are required parameters for a given BatchCluster.
 */
export interface ChildProcessFactory {
    /**
     * Factory function to spawn child processes.
     *
     * **CRITICAL**: If you spawn a child process and then reject the promise,
     * **YOU** are responsible for killing the spawned process. BatchCluster
     * cannot track processes that were never returned.
     *
     * Safe pattern:
     * ```typescript
     * async function myFactory(): Promise<ChildProcess> {
     *   const proc = spawn("my-command", args);
     *   try {
     *     await someValidation(proc);
     *     return proc;
     *   } catch (error) {
     *     proc.kill();  // REQUIRED: Clean up before rejecting!
     *     throw error;
     *   }
     * }
     * ```
     *
     * Unsafe pattern (LEAKS PROCESSES):
     * ```typescript
     * async function leakyFactory(): Promise<ChildProcess> {
     *   const proc = spawn("my-command", args);
     *   await someValidation(proc);  // If this throws, proc is orphaned!
     *   return proc;
     * }
     * ```
     */
    readonly processFactory: () => child_process.ChildProcess | Promise<child_process.ChildProcess>;
}
